/**
 *  Copyright (c) 2014-2015, Facebook, Inc.
 *  Copyright (c) 2016, Vishesh Yadav
 *
 *  All rights reserved.
 *
 *  This source code is licensed under the BSD-style license found in the
 *  LICENSE file in the root directory of this source tree. An additional grant
 *  of patent rights can be found in the PATENTS file in the same directory.
 *
 */

import { smi } from './math.js'

export function hashMix(hashes) {
    let result = 0;
    for (let h of hashes) {
	result += h;
	result += (result << 10);
	result ^= (result >> 6);
    }
    return result;
}

export function hash(o, getValue, callHashCode) {
    if (o === false || o === null || o === undefined) {
	return 0;
    }
    if (getValue === true && typeof o.valueOf === 'function') {
	o = o.valueOf();
	if (o === false || o === null || o === undefined) {
	    return 0;
	}
    }
    if (o === true) {
	return 1;
    }
    var type = typeof o;
    if (type === 'number') {
	if (o !== o || o === Infinity) {
	    return 0;
	}
	var h = o | 0;
	if (h !== o) {
	    h ^= o * 0xFFFFFFFF;
	}
	while (o > 0xFFFFFFFF) {
	    o /= 0xFFFFFFFF;
	    h ^= o;
	}
	return smi(h);
    }
    if (type === 'string') {
	return o.length > STRING_HASH_CACHE_MIN_STRLEN
	    ? cachedHashString(o)
	    : hashString(o);
    }
    if (callHashCode === true && typeof o.hashCode === 'function') {
	return o.hashCode();
    }
    if (type === 'object') {
	return hashJSObj(o);
    }
    if (typeof o.toString === 'function') {
	return hashString(o.toString());
    }
    throw new Error('Value type ' + type + ' cannot be hashed.');
}

function cachedHashString(string) {
    var hash = stringHashCache[string];
    if (hash === undefined) {
	hash = hashString(string);
	if (STRING_HASH_CACHE_SIZE === STRING_HASH_CACHE_MAX_SIZE) {
	    STRING_HASH_CACHE_SIZE = 0;
	    stringHashCache = {};
	}
	STRING_HASH_CACHE_SIZE++;
	stringHashCache[string] = hash;
    }
    return hash;
}

// http://jsperf.com/hashing-strings
export function hashString(string) {
    // This is the hash from JVM
    // The hash code for a string is computed as
    // s[0] * 31 ^ (n - 1) + s[1] * 31 ^ (n - 2) + ... + s[n - 1],
    // where s[i] is the ith character of the string and n is the length of
    // the string. We "mod" the result to make it between 0 (inclusive) and 2^31
    // (exclusive) by dropping high bits.
    var hash = 0;
    for (var ii = 0; ii < string.length; ii++) {
	hash = 31 * hash + string.charCodeAt(ii) | 0;
    }
    return smi(hash);
}

export function hashJSObj(obj) {
    var hash;
    if (usingWeakMap) {
	hash = weakMap.get(obj);
	if (hash !== undefined) {
	    return hash;
	}
    }

    hash = obj[UID_HASH_KEY];
    if (hash !== undefined) {
	return hash;
    }

    if (!canDefineProperty) {
	hash = obj.propertyIsEnumerable && obj.propertyIsEnumerable[UID_HASH_KEY];
	if (hash !== undefined) {
	    return hash;
	}

	hash = getIENodeHash(obj);
	if (hash !== undefined) {
	    return hash;
	}
    }

    hash = ++objHashUID;
    if (objHashUID & 0x40000000) {
	objHashUID = 0;
    }

    if (usingWeakMap) {
	weakMap.set(obj, hash);
    } else if (isExtensible !== undefined && isExtensible(obj) === false) {
	throw new Error('Non-extensible objects are not allowed as keys.');
    } else if (canDefineProperty) {
	Object.defineProperty(obj, UID_HASH_KEY, {
	    'enumerable': false,
	    'configurable': false,
	    'writable': false,
	    'value': hash
	});
    } else if (obj.propertyIsEnumerable !== undefined &&
               obj.propertyIsEnumerable === (obj.constructor.prototype
					     .propertyIsEnumerable)) {
	// Since we can't define a non-enumerable property on the object
	// we'll hijack one of the less-used non-enumerable properties to
	// save our hash on it. Since this is a function it will not show up in
	// `JSON.stringify` which is what we want.
	obj.propertyIsEnumerable = function() {
	    return this.constructor.prototype.propertyIsEnumerable
		.apply(this, arguments);
	};
	obj.propertyIsEnumerable[UID_HASH_KEY] = hash;
    } else if (obj.nodeType !== undefined) {
	// At this point we couldn't get the IE `uniqueID` to use as a hash
	// and we couldn't use a non-enumerable property to exploit the
	// dontEnum bug so we simply add the `UID_HASH_KEY` on the node
	// itself.
	obj[UID_HASH_KEY] = hash;
    } else {
	throw new Error('Unable to set a non-enumerable property on object.');
    }

    return hash;
}

// Get references to ES5 object methods.
var isExtensible = Object.isExtensible;

// True if Object.defineProperty works as expected. IE8 fails this test.
var canDefineProperty = (function() {
    try {
	Object.defineProperty({}, '@', {});
	return true;
    } catch (e) {
	return false;
    }
}());

// IE has a `uniqueID` property on DOM nodes. We can construct the hash from it
// and avoid memory leaks from the IE cloneNode bug.
function getIENodeHash(node) {
    if (node && node.nodeType > 0) {
	switch (node.nodeType) {
	case 1: // Element
            return node.uniqueID;
	case 9: // Document
            return node.documentElement && node.documentElement.uniqueID;
	}
    }
}

// If possible, use a WeakMap.
var usingWeakMap = typeof WeakMap === 'function';
var weakMap;
if (usingWeakMap) {
    weakMap = new WeakMap();
}

var objHashUID = 0;

var UID_HASH_KEY = '__immutablehash__';
if (typeof Symbol === 'function') {
    UID_HASH_KEY = Symbol(UID_HASH_KEY);
}

var STRING_HASH_CACHE_MIN_STRLEN = 16;
var STRING_HASH_CACHE_MAX_SIZE = 255;
var STRING_HASH_CACHE_SIZE = 0;
var stringHashCache = {};
