#lang racket

(require syntax/moddep
         graph
         threading
         "config.rkt"
         "util.rkt")

(provide follow-symbol
         get-export-tree
         get-module-deps
         module-deps/tsort
         module-deps/tsort-inv)

;; There are particularly two way's we can figure out what to export
;; and import in a module.
;;
;; 1. Parse require and provide specs. This although obvious and and
;;    likely preferred way, doesn't fit well with ES6 module
;;    system. In ES6 there is nothing like (all-defined-out) which
;;    exports all bindings in current namespace. This implies that
;;    we have to export each identifier individually, resulting in
;;    exponential growth of import/export statements.
;;
;; 2. Considering the drawback discussed and importance of generating
;;    compact code, we instead import each module as an object. With
;;    help of ExportTree and `identifier-binding` function we can
;;    follow each binding to its exact source.
;;
;;    Map generated by `make-module-name-map` will help us give a
;;    unique name to to each module object for ModulePath.

;; ExportTree is (Map ModulePath ExportOriginMap)
;;
;;   Map from a module path to list of all exports from that module,
;;   paired with origin of each of those exports.

;; ExportOriginMap is (Map Symbol (Maybe (Pairof ModulePath Symbol)))
;;
;;   Map between a Symbol which represents an exported identifier
;;   of module. The mapped value is either false if identifier is
;;   defined in current module, or (Pairof ModulePath Symbol) where
;;   car is path of module from which identifier is imported and
;;   cdr is name of this identifier in that module.

;; ExportTree ResolvedModulePath Symbol -> (Maybe (Pairof (Listof ModulePath) Symbol))
;; Track the modules from where id is imported starting from
;; src. We except id to be present in src. Only exports are
;; checked at each level, not imports. Returned value is a
;; a pair, where car is list of modules to follow and cdr is
;; the identifier name exported by module where it is defined
;; #f is returned if id is not exported.
(define (follow-symbol tree src id)
  (define (result src* id*)
    (cons (reverse src*) (first id*)))
  (let loop ([src* (list src)]
             [id* (list id)])
    (define module-exports (hash-ref tree (first src*) #f))
    (and module-exports
         (match (hash-ref module-exports (first id*) '())
           [(cons (? symbol? next-module) next-id)
            ;; When module name is a symbol such as #%kernel, #%unsafe ...
            (result (cons next-module src*)
                    (cons next-id id*))]
           [(cons next-module next-id)
            (loop (cons next-module src*)
                  (cons next-id id*))]
           [#f (result src* id*)]
           ['() #f]))))

;; ModulePath -> ExportTree
;; Return whole tree of exports with its source starting
;; from mod-name (ModulePath)
(define (get-export-tree mod-name)
  (define modules (filter-not symbol? (module-deps/tsort-inv
                                       (get-module-deps mod-name))))
  (for/hash ([m (append (set->list primitive-modules) modules)])
    (values m (get-exports/modpath m))))

;; ModulePath -> ExportOriginMap
;; For module at mod-path, returns a map of exported
;; identifiers with their corresponding origin module. If
;; identifier maps to #f, it is defined in mod-path itself.
(define (get-exports/modpath mod-path)
  (define-values (exports exported-syntax)
    ;; If its a primitive module, use the RacketScript implementation instead.
    (~> (resolve-module-path (actual-module-path mod-path) #f)
        (get-module-code _)
        (module-compiled-exports _)))
  (make-immutable-hash (parse-exports mod-path exports)))

;; ModulePath Exports -> ExportOriginMap
;; Takes in result of module->exports or module-compiled-exports
;; and returns an ExportOriginMap. mod-path is required to
;; resolve path indexes in exports
(define (parse-exports mod-path exports)
  ;; Takes a module exports entry in form (Pairof Symbol (Listof
  ;; ModPathIndex)) and return (list Symbol (Maybe ModPathIndex))
  (define (fix-entry e)
    (match e
      [(list id '()) (cons (first e) #f)]
      [(list id (list (? module-path-index? mod) ...))
       (cons id (cons (resolve-module-path-index (first mod) mod-path)
                      id))]
      [(list id (list
                 (list
                  (? module-path-index? mod) _ (? symbol? orig-name) _) ...))
       (cons id (cons (resolve-module-path-index (first mod) mod-path)
                      (first orig-name)))]))
  (for/fold ([r '()])
            ([i* exports])
    (match i*
      [(list 0 entry ...)
       (append r (map fix-entry entry))]
      [_ r])))

;; (Map Path (Listof Path)) -> (Listof Path)
;; Topologically sorted module dependency graph
(define (module-deps/tsort-inv mod-deps)
  (~> (hash->list mod-deps)
      (unweighted-graph/adj _)
      (tsort _)))
(define (module-deps/tsort mod-deps)
  (~> (hash->list mod-deps)
      (unweighted-graph/adj _)
      (transpose _)
      (tsort _)))

;; Path -> (Map Path (Listof Path))
;; Returns a adjecency map of module imports
(define (get-module-deps mod-path)
  (define graph (make-hash))
  (define (build-graph mod-path)
    (define path (resolve-module-path mod-path #f))
    (define code (get-module-code path))
    (define imports (module-compiled-imports code))
    (hash-set! graph path '())
    (for ([r imports])
      (match-define (cons i mods) r)
      (for ([mod mods])
        (match (resolve-module-path-index mod path)
          [#f (void)]
          [`(submod ,path ,mod) (void)]
          [`(submod ,path ,mod ,literal-sets) (void)]
          [(? symbol? b) (hash-set! graph b '())]
          [`,resolved-path (define new-mod (simplify-path resolved-path))
                           (hash-update! graph path (λ (v) (cons new-mod v)))
                           (unless (hash-ref graph new-mod #f)
                             (build-graph new-mod))]))))
  (build-graph mod-path)
  graph)
